<?php

/**
 * The Visitor Map code goes below.
 */


// Create visitor_locations table if it doesn't exist
function create_visitor_locations_table() {
    global $wpdb;
    
    $table_name = 'visitor_locations';
    $charset_collate = $wpdb->get_charset_collate();
    
    // Check if table exists
    if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
        $sql = "CREATE TABLE $table_name (
            id INT(11) NOT NULL AUTO_INCREMENT,
            ip_address VARCHAR(45) NOT NULL,
            latitude DECIMAL(10,8) NOT NULL,
            longitude DECIMAL(11,8) NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY unique_location (latitude, longitude)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Add indexes for better performance
        $wpdb->query("CREATE INDEX idx_latitude ON $table_name (latitude)");
        $wpdb->query("CREATE INDEX idx_longitude ON $table_name (longitude)");
        $wpdb->query("CREATE INDEX idx_ip_address ON $table_name (ip_address)");
    }
}
add_action('init', 'create_visitor_locations_table');

// Register visitor on every page load
function register_visitor() {
    // Don't run in admin area
    if (is_admin()) {
        return;
    }
    
    // Use WordPress database connection
    global $wpdb;
    
    // Your table name
    $table_name = 'visitor_locations';
    
    // Function to get visitor location
    if (!function_exists('getVisitorLocation')) {
        function getVisitorLocation($ip) {
            // Try free service first (no API key needed)
            $url = "http://ip-api.com/json/{$ip}";
            $response = @file_get_contents($url);
            
            if ($response) {
                $data = json_decode($response, true);
                if ($data['status'] == 'success') {
                    return [
                        'latitude' => $data['lat'],
                        'longitude' => $data['lon']
                    ];
                }
            }
            
            return null;
        }
    }
    
    // Get current visitor's IP
    $visitor_ip = $_SERVER['REMOTE_ADDR'];
    if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
        $visitor_ip = trim($ips[0]);
    }
    
    // Skip localhost/private IPs
    if ($visitor_ip == '127.0.0.1' || $visitor_ip == '::1' || substr($visitor_ip, 0, 7) == '192.168') {
        return;
    }
    
    // Get and save current visitor location
    $location = getVisitorLocation($visitor_ip);
    if ($location) {
        // Round coordinates to 4 decimal places (approx. 11 meters precision)
        // This prevents minor variations from creating duplicate locations
        $rounded_lat = round($location['latitude'], 4);
        $rounded_lon = round($location['longitude'], 4);
        
        // Check if location already exists (based on rounded coordinates)
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table_name 
             WHERE ROUND(latitude, 4) = %f 
             AND ROUND(longitude, 4) = %f",
            $rounded_lat,
            $rounded_lon
        ));
        
        // Only insert if location doesn't exist yet
        if (empty($existing)) {
            $wpdb->insert(
                $table_name,
                array(
                    'ip_address' => $visitor_ip,
                    'latitude' => $location['latitude'],
                    'longitude' => $location['longitude']
                ),
                array('%s', '%f', '%f')
            );
        } else {
            // Optional: Update IP address for existing location
            // This keeps the most recent IP for that location
            $wpdb->update(
                $table_name,
                array('ip_address' => $visitor_ip),
                array('id' => $existing),
                array('%s'),
                array('%d')
            );
        }
    }
}
// Run on every page load
add_action('wp', 'register_visitor');

// Display the map only where shortcode is used
function display_visitor_map() {
    // Don't run in admin area
    if (is_admin()) {
        return '';
    }
    
    // Use WordPress database connection
    global $wpdb;
    
    // Your table name
    $table_name = 'visitor_locations';
    
    // Get all visitor locations
    $locations = $wpdb->get_results("SELECT latitude, longitude FROM $table_name ORDER BY created_at DESC", ARRAY_A);
    
    // Generate the map HTML
    ob_start();
 ?>
    
    <div class="visitor-map-container" style="width: 100%; max-width: 760px; margin: 30px auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1);">
        <h3 style="text-align: center; margin-bottom: 20px; color: #333;">Visitor Locations Around the World</h3>
        <div id="simple-visitor-map" style="width: 100%; height: 400px; border: 1px solid #ddd; border-radius: 4px;"></div>
        <p style="text-align: center; color: #666; margin-top: 15px; font-size: 14px;">
            Total unique visitor locations: <?php echo count($locations); ?>
        </p>
    </div>
    
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.7.1/dist/leaflet.css" />
    <script src="https://unpkg.com/leaflet@1.7.1/dist/leaflet.js"></script>
    
    <style>
    #simple-visitor-map {
        pointer-events: none;
        cursor: default;
    }
    .leaflet-container {
        background: #e8f4f8 !important;
        cursor: default !important;
    }
    .leaflet-control-container {
        display: none !important;
    }
    </style>
    
<script>
document.addEventListener('DOMContentLoaded', function() {
    const locations = <?php echo json_encode($locations); ?>;
    
    // Check if we have locations
    if (locations.length === 0) {
        document.getElementById('simple-visitor-map').innerHTML = '<p style="text-align: center; padding: 50px;">No visitor data available yet.</p>';
        return;
    }
    
    // Create simple non-interactive map
    const map = L.map('simple-visitor-map', {
        center: [20, 0], // Slightly north to center populated areas
        zoom: 1.5, // Better world view
        minZoom: 1,
        maxZoom: 1.5, // Prevent zooming in too much
        zoomControl: false,
        attributionControl: false,
        dragging: false,
        scrollWheelZoom: false,
        doubleClickZoom: false,
        boxZoom: false,
        keyboard: false,
        tap: false,
        touchZoom: false,
        // Limit vertical view to exclude polar regions
        maxBounds: [[-60, -180], [85, 180]] // Leaves out Antarctica and far north
    });
    
    // Add map tiles
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors',
        interactive: false,
        noWrap: false // Allow world wrapping
    }).addTo(map);
    
    // Create a custom icon for 2px dots
    const dotIcon = L.divIcon({
        html: '<div style="width: 2px; height: 2px; background: red; border-radius: 50%; opacity: 0.7;"></div>',
        className: 'tiny-dot',
        iconSize: [2, 2],
        iconAnchor: [1, 1]
    });
    
    // Add visitor locations as 2px red dots
    locations.forEach(location => {
        const lat = parseFloat(location.latitude);
        const lon = parseFloat(location.longitude);
        
        if (!isNaN(lat) && !isNaN(lon) && lat !== 0 && lon !== 0) {
            // Filter out extreme latitudes if needed
            if (lat >= -60 && lat <= 85) { // Within our view bounds
                L.marker([lat, lon], {
                    icon: dotIcon,
                    interactive: false
                }).addTo(map);
            }
        }
    });
    
    // Disable all interactions
    map.dragging.disable();
    map.touchZoom.disable();
    map.doubleClickZoom.disable();
    map.scrollWheelZoom.disable();
    
    // Ensure map shows full width
    setTimeout(() => {
        map.invalidateSize();
        // Force full width display
        map.setView([20, 0], 1.5);
    }, 100);
});
</script>

   <?php
    return ob_get_clean();
}

// Register the shortcode
add_shortcode('visitor_map', 'display_visitor_map');

// Function to migrate existing data if needed (run once manually)
function migrate_to_location_based() {
    global $wpdb;
    $table_name = 'visitor_locations';
    
    // Get all duplicate IPs and keep only one per location
    $wpdb->query("
        CREATE TEMPORARY TABLE temp_keep_locations AS
        SELECT MIN(id) as keep_id, 
               ROUND(latitude, 4) as lat_rounded, 
               ROUND(longitude, 4) as lon_rounded
        FROM $table_name 
        GROUP BY lat_rounded, lon_rounded
    ");
    
    // Delete duplicates
    $wpdb->query("
        DELETE FROM $table_name 
        WHERE id NOT IN (SELECT keep_id FROM temp_keep_locations)
    ");
    
    // Drop temp table
    $wpdb->query("DROP TEMPORARY TABLE temp_keep_locations");
}
// Uncomment to run migration once, then comment out again
// add_action('init', 'migrate_to_location_based');
?>